//gcc zip_sha256.c -o zip_sha256 -lzip sha2.c sha2.h

#include <zip.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include "sha2.h"

void hash_file(SHA256_CTX *hash, zip_file_t *readfile, uint64_t size);
int cmpfunc (const void * a, const void * b);


int main(int argc, char *argv[]){
  int errorp;
  if(argc<2){
    printf("Usage: zip_sha256 <input.zip> [input.zip] ...\n   or: zip_sha256 --sum [input.zip]\n"); //It's a pun: Short for 'summary' as well as Latin for 'I am.' Both are valid meanings.
    printf("       In standard mode, prints the sha256 of all files in the zip (In consistent order - usually asciibetical, unicode will do 'something' odd but consistent.)\n");
    printf("       In sum mode, will print the sha256 of all files concatenated, in the same order.\n       --sum2 will do likewise, but also concatenate filenames.\n");
    exit(1);
  }
  int omnimode=0;
  int withnames=0;
  if(!strcmp(argv[1], "--sum")){
    omnimode=1;
    if(argc>3){
      printf("Usage: zip_sha256 <input.zip> [input.zip] ...\n   or: zip_sha256 --sum [input.zip]\n");
      exit(1);
    }
  }
  if(!strcmp(argv[1], "--sum2")){
    omnimode=1;
    withnames=1;
    if(argc>3){
      printf("Usage: zip_sha256 <input.zip> [input.zip] ...\n   or: zip_sha256 --sum [input.zip]\n");
      exit(1);
    }
  }
  SHA256_CTX omnihash;
  if(omnimode)
    SHA256_Init(&omnihash);

  uint32_t infile_num;
  for(infile_num=omnimode+1; infile_num<argc;infile_num++){
    zip_t *zipfile=zip_open(argv[infile_num], ZIP_RDONLY, &errorp);
    if(!zipfile){
      fprintf(stderr,"zip_sha256 failed opening input file.\n");
      exit(1);
    }

    uint64_t numfiles=zip_get_num_entries(zipfile, 0);
//    fprintf(stderr, "File %s entries %lu.\n", argv[infile_num], numfiles);
//    struct zip_stat innerfiles[numfiles];
    struct zip_stat *innerfiles=malloc(sizeof(struct zip_stat)*numfiles);
//    memset(innerfiles, sizeof(struct zip_stat)*numfiles, 0);
    uint64_t n;
    for(n=0;n<numfiles;n++)
      zip_stat_index(zipfile, n, 0,&innerfiles[n]);

   qsort(innerfiles, numfiles, sizeof(struct zip_stat), cmpfunc);

    for(n=0;n<numfiles;n++){
      struct zip_stat filestat=innerfiles[n];
//      memcpy(&filestat, &innerfiles[n], sizeof(struct zip_stat));
//      printf("%s,%lu,%lu\n", filestat.name, filestat.size,filestat.index);
      zip_file_t *readfile=zip_fopen_index(zipfile, filestat.index, 0);
      if(!readfile){
        fprintf(stderr,"Fatal error in zip_sha256.\n");
        exit(1);
      }
      if(omnimode){
        hash_file(&omnihash, readfile, filestat.size);
        if(withnames)
          SHA256_Update(&omnihash, filestat.name, strlen(filestat.name));
      }else{
        uint8_t hash[32];
        SHA256_CTX sha256er;
        SHA256_Init(&sha256er);
        hash_file(&sha256er, readfile, filestat.size);
        SHA256_Final(hash, &sha256er);
        int m;
        for(m=0;m<32;m++)
          printf("%02X", hash[m]);
        printf("\n");
      }
      zip_fclose(readfile);
    }
    zip_close(zipfile);
  }
  if(omnimode){
    uint8_t hash[32];
    SHA256_Final(hash, &omnihash);
    int m;
    for(m=0;m<32;m++)
      printf("%02X", hash[m]);
    printf("\n");
  }
}

void hash_file(SHA256_CTX *hash, zip_file_t *readfile, uint64_t size){
  char temp[512*1024];
  uint64_t read=0;
  zip_int64_t ret=0;
  do{
    ret=zip_fread(readfile, temp, 512*1024);
    if(ret== -1)
      exit(1);
    SHA256_Update(hash, temp, ret);
    read+=ret;
  }while(ret);
}

int cmpfunc (const void * a, const void * b) {
  const struct zip_stat *structa=a;
  const struct zip_stat *structb=b;
  return(strcmp(structa->name, structb->name));
}
