#define HDLC_CSUM_NONE 0//Don't check the checksum.
#define HDLC_CSUM_AX25 1//CRC16, but unpleasantly computed in LSB-first manner. Good for hardware implimentation.
#define HDLC_CSUM_CRC32B 2//This is not the same as the original HDLC CRC32. That particular sum is hardly ever used now.
                         //This one is pretty common though, and easier to impliment in software.
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

class hdlc_composer{

  public:
    uint16_t preamble_len; //This isn't actually used. Not directly within the object.
    uint8_t *frame;
    uint32_t frame_length; //Length of frame payload. DOES NOT INCLUDE CRC!
    uint8_t checksum; //Type of checksum to use, if any.

    bool no_diff_encode; //Bypasses the differential decoder, if your physical layer doesn't need it.
    void (* pre_send_function)(hdlc_composer*); //If set, this function will be called after the checksum but before sending.
                                       //It is intended as a location in which to add application-specific FEC information.

    void set_output_function(int (* new_output_function)(bool)){
      output_function=new_output_function;
    }
    hdlc_composer(uint32_t max_size){
      max_frame_size=max_size;
      frame=(uint8_t*)malloc(max_frame_size);
      stuffing_counter=1;
      checksum=HDLC_CSUM_CRC32B;
      preamble_len=100;
      output_function=NULL;no_diff_encode=false;pre_send_function=NULL;frame_length=0;diffbit=0;
    }
    hdlc_composer(){};

    ~hdlc_composer(){
      free(frame);
    }
  void send_frame(){
    if(checksum==HDLC_CSUM_CRC32B){
      uint32_t crc=crc32b(frame,frame_length);
      frame[frame_length  ]=((crc   )&0xFF); //This method of copying ensures that the byte ordering will be the same
      frame[frame_length+1]=((crc>>8)&0xFF); //regardless of the processor architecture. Endianness issues thus eliminated.
      frame[frame_length+2]=((crc>>16)&0xFF);
      frame[frame_length+3]=((crc>>24)&0xFF);
      frame_length+=4; //CRC, check!*/
    }
    if(pre_send_function)
      pre_send_function(this);
    while(send_frame_2());
      

  }
  int send_frame_2(){
    uint8_t e;
    for(unsigned int n=0;n<frame_length;n++){
      uint8_t val=frame[n];
      e=0;
      e=e|sendbit((val>>0)&0x01);
      e=e|sendbit((val>>1)&0x01);
      e=e|sendbit((val>>2)&0x01);
      e=e|sendbit((val>>3)&0x01);
      e=e|sendbit((val>>4)&0x01);
      e=e|sendbit((val>>5)&0x01);
      e=e|sendbit((val>>6)&0x01);
      e=e|sendbit((val>>7)&0x01);
      if(e){
        fprintf(stderr, "Error sending frame.\n");fflush(stderr);
        send_sync_sequence(10);
        return(1);
      }
    }
    send_sync_sequence(1);
    return(0);
  }
  void send_sync_sequence(int repetitions){
    //This bit pattern is based upon HDLC, a very old framing method.
    //It's used in conjunction with bit stuffing.
    int n;
    if(!stuffing_counter)
      return; //There's no point sending this if stuffing is disabled.
    stuffing_counter=0;
    for(n=0;n<repetitions;n++){
      sendbit(0);sendbit(1);sendbit(1);sendbit(1);
      sendbit(1);sendbit(1);sendbit(1);sendbit(0);stuffing_counter=0;
    }
    stuffing_counter=1;
  }

    uint32_t max_frame_size;
  uint8_t stuffing_counter;

  private:
    int (* output_function)(bool); //This gets called for every *bit* of output. Fair chunk of overhead!
                                   //If it returns -1, the frame send will be aborted and retried.

    static uint32_t crc32b(uint8_t *message,uint32_t len) {
      int i=0,j;
      uint32_t byte, crc, mask;

      crc = 0xFFFFFFFF;
      while (i!=len) {
        byte = message[i++];            // Get next byte.
        crc = crc ^ byte;
        for (j = 7; j >= 0; j--) {    // Do eight times.
          mask = -(crc & 1);
          crc = (crc >> 1) ^ (0xEDB88320 & mask);
        }
      }
      return ~crc;
    }

  bool diffbit; //Remember, differential encoding. A 0 is a change, a 1 is a repeat.

  virtual int sendbit(bool bit){

    //Stuffing first - we need to make sure that five consecutive ones are followed by a zero.
    //Why? Two reasons: One, to make sure the 01111110 sync flag is distinctive. Look up HDLC for details.
    //Two, because occasional phase changes actually help the PLL at the receiving end aquire lock.
      if(stuffing_counter){ //Setting stuffing_counter to 0 is used to disable stuffing while sending the sync symbol.
        if(bit)
          stuffing_counter++;
        if(stuffing_counter==6){
          stuffing_counter=1;
          if(sendbit(1)) return(1);bit=0;
        }
        if(!bit)
          stuffing_counter=1;
      }

    if(no_diff_encode)
      return(output_function(bit));
    if(!bit)
      diffbit=!diffbit;
    return(output_function(diffbit));
  }

};
class hdlc_receiver{
//This object searches within a differential-coded bit stream for frame encoded using a format similar to, but not quite
//  the same as, HDLC. It's designed for, among other things, AX.25 frames - it's a ham radio protocol, and a distant
//  descendent of a protocol based upon HDLC. It's not HDLC, but it shares the same design at the very lowest level.

//This object handles the differential encoding, bit stuffing, and identification of a valid frame.
//It DOES NOT handle anything to do with AX.25. That's another layer.

  public:
    uint8_t *frame;
    uint32_t frame_length;
    uint8_t checksum; //Type of checksum to use, if any.
    //Field max/min length fields are the raw, on-the-wire length - including checksum and any FEC bytes.
    uint32_t max_length; //Reject anything longer than this...
    uint32_t min_length; //... or shorter than this.

    bool no_diff_encode; //Bypasses the differential decoder, if your physical layer doesn't need it.
    void (* post_receive_function)(hdlc_receiver*); //If set, this function will be called before verifying the checksum.
                                       //It is intended as a location in which to process application-specific FEC information.


    void new_bit(bool new_bit){

      uint8_t decbit=1;
      if(no_diff_encode)
        decbit=new_bit;
      else{
        if(new_bit!=lastbit)
          decbit=0;
        lastbit=new_bit;
      }
      syncdetect=(syncdetect<<1)|decbit;
      if(syncdetect==0x7E){
        if(!err_locked && ((bitcounter&0x07)==6) &&
          frame_length>=min_length&&
          frame_length<=max_length){
//printf("Potential frame, len %u chk %u.\n", frame_length,verify_checksum());fflush(stdout);
            if(post_receive_function)
              post_receive_function(this);

            if(verify_checksum())
              potential_frame();
          }
        bitcounter=0;frame_length=0;syncdetect=0;consecutive_ones=0;err_locked=false;
        return;
      }

      if(err_locked)
        return;
      if(consecutive_ones==5){
        consecutive_ones=0;
        return;
      }
      if(decbit)
        consecutive_ones++;
      else
        consecutive_ones=0;
      shiftregister=shiftregister>>1;//HDLC is LSb-first.
      if(decbit)
        shiftregister=shiftregister|0x80;

      if((bitcounter&0x07)==0x07){
        frame[frame_length++]=shiftregister;
        if(frame_length>max_frame_size || frame_length>max_length)
          err_locked=true;
      }
      bitcounter++;

    }
    void set_callback(void (* newGotFrameCallback)(hdlc_receiver*)){
      gotFrameCallback=newGotFrameCallback;
    }
    hdlc_receiver(){
    }

    hdlc_receiver(uint32_t max_size){
      max_frame_size=max_size;
      frame=(uint8_t*)malloc(max_frame_size+1);
      min_length=1;max_length=max_size;
      checksum=HDLC_CSUM_CRC32B;
      err_locked=false;
      gotFrameCallback=NULL;post_receive_function=NULL;no_diff_encode=false;
      frame_length=0;
    }
    ~hdlc_receiver(){
      free(frame);
    }
    void (* gotFrameCallback)(hdlc_receiver*);
    bool verify_checksum(){
      if(checksum==HDLC_CSUM_NONE)
        return(true);
      if(checksum==HDLC_CSUM_CRC32B){
        if(frame_length<5)
          return(false);
        uint32_t crc=crc32b(frame,frame_length-4);
        uint32_t targetcrc=frame[frame_length-4];//All this shuffling around is to make sure behavior isn't endian.
        targetcrc=targetcrc|frame[frame_length-3]<<8;
        targetcrc=targetcrc|frame[frame_length-2]<<16;
        targetcrc=targetcrc|frame[frame_length-1]<<24;
        return(crc==targetcrc);
      }
    }
  private:
    bool lastbit;
    uint32_t max_frame_size;
    uint8_t consecutive_ones; //For de-padding.
    uint8_t syncdetect;
    uint8_t shiftregister;
    uint8_t bitcounter;
    bool err_locked; //An irrecoverable error occured while decoding this frame. Cease decoding until next frame marker.

    virtual void potential_frame(){ //This stub-function exists only for convenience of overriding in subclasses.
      gotFrameCallback(this);
    }


    static uint32_t crc32b(uint8_t *message,uint32_t len) {
      int i=0,j;
      uint32_t byte, crc, mask;

      crc = 0xFFFFFFFF;
      while (i!=len) {
        byte = message[i++];            // Get next byte.
        crc = crc ^ byte;
        for (j = 7; j >= 0; j--) {    // Do eight times.
          mask = -(crc & 1);
          crc = (crc >> 1) ^ (0xEDB88320 & mask);
        }
      }
      return ~crc;
    }
};
